# Module 5: Functions, Vectors & Iteration
# Practice applying R4DS concepts to clinical programming

# ===========================
# SETUP: Load Required Packages
# ===========================

library(dplyr)
library(tibble)
library(lubridate)
library(stringr)
library(purrr)

cat("=== Module 5 Exercise: Functions, Vectors & Iteration ===\n")
cat("Based on R4DS Chapters 19, 20, and 21\n")
cat("Practice applying these concepts to clinical programming\n\n")

# ===========================
# EXERCISE 1: Functions (R4DS Chapter 19)
# ===========================

cat("=== Exercise 1: Functions (R4DS Chapter 19) ===\n")

# Create sample clinical data
demo <- tibble(
  USUBJID = paste0("001-", sprintf("%03d", 1:10)),
  AGE = c(23, 45, 67, 52, 71, 34, 58, 63, 29, 76),
  SEX = c("F", "M", "F", "M", "F", "F", "M", "F", "M", "F"),
  WEIGHT = c(65, 80, 58, 75, 62, 70, 85, 60, 78, 55),
  HEIGHT = c(160, 175, 155, 180, 158, 165, 185, 162, 172, 150),
  RFSTDTC = "2024-01-15"
)

print("Sample clinical data:")
print(demo)

# TASK 1.1: Apply the DRY principle
# Instead of repeating normalization code, create a function

# DON'T DO THIS (repetitive):
# demo$WEIGHT_NORM <- (demo$WEIGHT - min(demo$WEIGHT, na.rm = TRUE)) / (max(demo$WEIGHT, na.rm = TRUE) - min(demo$WEIGHT, na.rm = TRUE))
# demo$HEIGHT_NORM <- (demo$HEIGHT - min(demo$HEIGHT, na.rm = TRUE)) / (max(demo$HEIGHT, na.rm = TRUE) - min(demo$HEIGHT, na.rm = TRUE))

# YOUR TASK: Create a rescale function to normalize values to 0-1 range
rescale01 <- function(x) {
  # YOUR CODE HERE
  # Hint: (x - min(x)) / (max(x) - min(x))
  # Don't forget to handle NA values
}

# Test your rescale function
test_values <- c(10, 20, 30, NA, 40)
cat("Original values:", test_values, "\n")
cat("Rescaled values:", rescale01(test_values), "\n")

# TASK 1.2: Function components and arguments
# Create a clinical age categorization function with proper arguments

clinical_age_category <- function(age, pediatric_cutoff = 18, elderly_cutoff = 65) {
  # YOUR CODE HERE
  # Use case_when() to create categories:
  # - NA ages should return "Unknown"
  # - age < pediatric_cutoff should return "Pediatric"
  # - age >= pediatric_cutoff & age < elderly_cutoff should return "Adult"
  # - age >= elderly_cutoff should return "Elderly"
}

# Test with different cutoffs
test_ages <- c(16, 25, 45, 67, 72, NA)
cat("\nAge categories (default cutoffs):\n")
print(clinical_age_category(test_ages))

cat("Age categories (pediatric: 21, elderly: 60):\n")
print(clinical_age_category(test_ages, pediatric_cutoff = 21, elderly_cutoff = 60))

# TASK 1.3: Function with ... (dot-dot-dot) argument
# Create a flexible summary function for clinical variables

clinical_summary <- function(data, ..., na.rm = TRUE, digits = 2) {
  # YOUR CODE HERE
  # Use summarise() and across() to calculate mean, sd, min, max for specified columns
  # Hint: across(c(...), list(mean = ~mean(.x, na.rm = na.rm), ...))
}

# Test your summary function
cat("\nClinical summary for AGE and WEIGHT:\n")
clinical_summary(demo, AGE, WEIGHT)

# TASK 1.4: Create BMI function with input validation
create_bmi_with_validation <- function(data, weight_var, height_var) {
  # YOUR CODE HERE
  # 1. Check if data is a data frame (use is.data.frame())
  # 2. Calculate BMI = weight / (height/100)^2
  # 3. Add BMI categories: "Underweight" (<18.5), "Normal" (18.5-24.9), "Overweight" (25-29.9), "Obese" (≥30)
  # 4. Handle missing values appropriately
}

# Test your BMI function
demo_with_bmi <- demo %>%
  create_bmi_with_validation(WEIGHT, HEIGHT)

cat("\nBMI results:\n")
print(demo_with_bmi %>% select(USUBJID, WEIGHT, HEIGHT, BMI, BMI_CATEGORY))

# ===========================
# EXERCISE 2: Vectors (R4DS Chapter 20)
# ===========================

cat("\n=== Exercise 2: Vectors (R4DS Chapter 20) ===\n")

# TASK 2.1: Understanding vector types in clinical data
cat("Understanding vector types:\n")

# Create vectors of different types commonly used in clinical data
logical_response <- c(TRUE, FALSE, TRUE, FALSE)     # Treatment response
visit_numbers <- c(1L, 2L, 3L, 4L)                 # Visit numbers (integer)
lab_values <- c(120.5, 85.2, 95.8, 110.1)         # Lab values (double)
subject_sex <- c("M", "F", "M", "F")               # Gender (character)

# YOUR TASK: Use typeof() to check the type of each vector
cat("Response vector type:", "YOUR CODE HERE", "\n")
cat("Visit numbers type:", "YOUR CODE HERE", "\n")
cat("Lab values type:", "YOUR CODE HERE", "\n")
cat("Subject sex type:", "YOUR CODE HERE", "\n")

# TASK 2.2: Vector coercion - handling messy clinical data
messy_lab_data <- c("120", "85", "normal", "95", "high", "75.5")

# YOUR TASK: Create a safe conversion function
safe_lab_conversion <- function(x) {
  # YOUR CODE HERE
  # 1. Replace "normal", "high", "low" with NA_character_
  # 2. Convert to numeric using as.numeric()
  # 3. Use suppressWarnings() to avoid warning messages
  # 4. Return the cleaned numeric vector
}

cleaned_labs <- safe_lab_conversion(messy_lab_data)
cat("Original messy data:\n")
print(messy_lab_data)
cat("Cleaned numeric data:\n")
print(cleaned_labs)

# TASK 2.3: Vector subsetting for clinical data analysis
subject_ids <- c("001-001", "001-002", "001-003", "001-004", "001-005")
ages <- c(25, 67, 45, 72, 34)
adverse_events <- c(2, 0, 1, 3, 1)

# Name the vectors for easier subsetting
names(ages) <- subject_ids
names(adverse_events) <- subject_ids

# YOUR TASKS:
# 1. Select subjects 1, 3, and 5 by position
selected_subjects_pos <- subject_ids[c(1, 3, 5)]  # Example provided

# 2. Select all subjects except the 2nd and 4th
selected_subjects_neg <- "YOUR CODE HERE"

# 3. Select elderly subjects (age >= 65) using logical subsetting
elderly_subjects <- "YOUR CODE HERE"

# 4. Select subjects with multiple adverse events (>1)
multiple_ae_subjects <- "YOUR CODE HERE"

# 5. Get ages for specific subjects by name
specific_ages <- ages[c("001-001", "001-004")]  # Example provided

cat("Subjects 1, 3, 5:", selected_subjects_pos, "\n")
cat("Elderly subjects:", elderly_subjects, "\n")
cat("Subjects with multiple AEs:", multiple_ae_subjects, "\n")

# TASK 2.4: Create clinical data with proper vector types
clinical_data <- tibble(
  USUBJID = paste0("SUB-", sprintf("%03d", 1:8)),
  AGE = c(25, 45, 67, 52, 71, 34, 58, 63),
  SEX = c("F", "M", "F", "M", "F", "F", "M", "F"),
  WEIGHT = c(65.5, 80.2, 58.7, 75.1, 62.3, 70.8, 85.4, 60.2),
  HEIGHT = c(160L, 175L, 155L, 180L, 158L, 165L, 185L, 162L),  # Note: integers
  VISIT = factor(c("Baseline", "Week 2", "Week 4", "Week 8", "Baseline", "Week 2", "Week 4", "Week 8")),
  TREATMENT_RESPONSE = c(TRUE, FALSE, TRUE, TRUE, FALSE, TRUE, FALSE, TRUE)
)

# YOUR TASK: Check the types of all columns
cat("\nColumn types in clinical data:\n")
print("YOUR CODE HERE")  # Hint: use sapply(clinical_data, typeof)
# ===========================
# EXERCISE 3: Iteration (R4DS Chapter 21)
# ===========================

cat("\n=== Exercise 3: Iteration (R4DS Chapter 21) ===\n")

# Create adverse events data for iteration exercises
ae_data <- tibble(
  USUBJID = c("001-001", "001-001", "001-002", "001-002", "001-003", "001-003"),
  AESEQ = c(1, 2, 1, 2, 1, 2),
  AEDECOD = c("  headache ", "NAUSEA", "fatigue", "  DIZZINESS  ", "rash", "COUGH"),
  CMDECOD = c("aspirin  ", "  IBUPROFEN", "acetaminophen  ", "NSAID", "  topical cream", "cough syrup"),
  AESTDTC = c("2024-01-20", "2024-01-25", "2024-01-18", "2024-01-22", "2024-01-26", "2024-01-28"),
  AEENDTC = c("2024-01-22", "2024-01-26", "2024-01-20", "2024-01-23", "2024-01-28", "2024-01-30"),
  RFSTDTC = c("2024-01-15", "2024-01-15", "2024-01-16", "2024-01-16", "2024-01-15", "2024-01-15")
)

print("AE data for iteration exercises:")
print(ae_data)

# TASK 3.1: For loops for summary statistics
numeric_vars <- c("AGE", "WEIGHT", "HEIGHT")

# YOUR TASK: Use a for loop to calculate summary statistics
# Pre-allocate results vector (important for performance!)
summary_results <- vector("list", length(numeric_vars))
names(summary_results) <- numeric_vars

for (i in seq_along(numeric_vars)) {
  # YOUR CODE HERE
  # For each numeric variable in clinical_data:
  # 1. Extract the variable data: var_data <- clinical_data[[numeric_vars[[i]]]]
  # 2. Calculate: n, mean, sd, min, max (handle NA values)
  # 3. Store results in summary_results[[i]]
}

cat("Summary statistics using for loop:\n")
print(summary_results)

# TASK 3.2: Modify existing objects with for loop
# Clean multiple text columns in ae_data
text_cols <- c("AEDECOD", "CMDECOD")

# YOUR TASK: Use for loop to standardize text columns
# For each column in text_cols:
# 1. Trim whitespace (str_trim)
# 2. Convert to uppercase (str_to_upper)
# 3. Replace multiple spaces with single space (str_replace_all("\\s+", " "))

for (col in text_cols) {
  # YOUR CODE HERE
}

cat("Cleaned AE data:\n")
print(ae_data)

# TASK 3.3: While loop for dose escalation simulation
dose_escalation <- function(starting_dose = 10, max_dose = 80, safety_prob = 0.8) {
  current_dose <- starting_dose
  doses <- current_dose
  step <- 1

  cat("Dose escalation simulation:\n")
  cat("Step", step, ": Starting dose =", current_dose, "mg\n")

  # YOUR TASK: Complete the while loop
  while ("YOUR CONDITION HERE") {
    # 1. Simulate safety assessment: safety_ok <- rbinom(1, 1, prob = safety_prob)
    # 2. If safe, escalate dose by 50% (but don't exceed max_dose)
    # 3. If not safe, break the loop
    # 4. Update step counter and doses vector
    # 5. Add safety check to prevent infinite loop (max 10 steps)

    # YOUR CODE HERE
  }

  return(doses)
}

# Test your dose escalation function
final_doses <- dose_escalation()

# TASK 3.4: Compare for loop vs functional programming
# Calculate the same summary statistics using map()

# YOUR TASK: Use map_dfr() to calculate summary statistics
summary_functional <- clinical_data %>%
  select(where(is.numeric)) %>%
  map_dfr("YOUR CODE HERE", .id = "variable")  # Complete the map_dfr call

cat("Summary using functional programming:\n")
print(summary_functional)

# Which approach do you prefer and why?

# ===========================
# EXERCISE 4: SAS Macro Translation with R4DS Concepts
# ===========================

cat("\n=== Exercise 4: SAS Macro Translation ===\n")

# TASK 4.1: Translate SAS macro to R function with proper vector handling
# SAS Macro example:
# %macro derive_studyday(indata=, outdata=, eventdt=, refdt=, studyday=);
#   data &outdata;
#     set &indata;
#     if not missing(&eventdt) and not missing(&refdt) then do;
#       if &eventdt >= &refdt then &studyday = &eventdt - &refdt + 1;
#       else &studyday = &eventdt - &refdt;
#     end;
#   run;
# %mend derive_studyday;

derive_study_day <- function(data, event_date_var, ref_date_var, new_var_name = "STUDY_DAY") {
  # YOUR CODE HERE
  # 1. Use mutate() and case_when()
  # 2. Convert dates using ymd() from lubridate
  # 3. Handle missing dates (return NA_real_)
  # 4. Apply the study day logic:
  #    - If event >= ref: study_day = event - ref + 1
  #    - If event < ref: study_day = event - ref
  # 5. Use !!new_var_name := to create the new column dynamically
}

# Test your function
ae_with_studyday <- ae_data %>%
  derive_study_day(AESTDTC, RFSTDTC, "AESTDY")

cat("AE data with study day:\n")
print(ae_with_studyday %>% select(USUBJID, AESTDTC, RFSTDTC, AESTDY))

# TASK 4.2: Create a comprehensive AE processing function with error handling
process_ae_data_complete <- function(data, ae_term_var, start_date_var, end_date_var, ref_date_var) {
  # YOUR CODE HERE
  # 1. Input validation:
  #    - Check if data is a data frame
  #    - Check if required columns exist
  # 2. Standardize AE terms (trim, uppercase)
  # 3. Calculate start and end study days
  # 4. Calculate duration in days
  # 5. Add validation flags for missing data
  # 6. Include appropriate error handling and warnings
}

# Test your comprehensive function
processed_ae <- ae_data %>%
  process_ae_data_complete(AEDECOD, AESTDTC, AEENDTC, RFSTDTC)

cat("Fully processed AE data:\n")
print(processed_ae)

# ===========================
# EXERCISE 5: Advanced Functional Programming with map()
# ===========================

cat("\n=== Exercise 5: Advanced Functional Programming ===\n")

# Create multiple study datasets for map() exercises
study_datasets <- list(
  study_a = tibble(
    USUBJID = c("A-001", "A-002", "A-003", "A-004"),
    AGE = c(45, 67, 34, 52),
    WEIGHT = c(70, 65, 75, 80),
    SEX = c("M", "F", "M", "F")
  ),
  study_b = tibble(
    USUBJID = c("B-001", "B-002", "B-003"),
    AGE = c(28, 49, 63),
    WEIGHT = c(68, 72, 58),
    SEX = c("F", "M", "F")
  ),
  study_c = tibble(
    USUBJID = c("C-001", "C-002", "C-003", "C-004", "C-005"),
    AGE = c(39, 58, 46, 62, 33),
    WEIGHT = c(74, 69, 77, 61, 73),
    SEX = c("M", "F", "M", "F", "M")
  )
)

# TASK 5.1: Basic map() functions with different return types
# Calculate mean age for each study using different map functions

# YOUR TASKS:
mean_ages_list <- study_datasets %>%
  map("YOUR CODE HERE")  # Returns list

mean_ages_vector <- study_datasets %>%
  map_dbl("YOUR CODE HERE")  # Returns numeric vector

age_summaries <- study_datasets %>%
  map_chr("YOUR CODE HERE")  # Returns character vector with formatted summaries

cat("Mean ages (list):\n")
print(mean_ages_list)
cat("Mean ages (vector):\n")
print(mean_ages_vector)
cat("Age summaries:\n")
print(age_summaries)

# TASK 5.2: map2() for two inputs
# Calculate BMI using map2_dbl()
weights <- c(70, 65, 80, 75, 68)
heights <- c(175, 160, 185, 180, 155)

bmis <- map2_dbl("YOUR CODE HERE")  # Complete the map2_dbl call
cat("BMI calculations:\n")
print(round(bmis, 2))

# TASK 5.3: pmap() for multiple inputs
patient_profiles <- list(
  weight = c(70, 65, 80, 75),
  height = c(175, 160, 185, 180),
  age = c(45, 62, 38, 55),
  sex = c("M", "F", "M", "F")
)

# Create a function that uses all inputs
calculate_risk_score <- function(weight, height, age, sex) {
  bmi <- weight / (height / 100)^2
  age_risk <- ifelse(age > 60, 1.2, 1.0)
  sex_risk <- ifelse(sex == "M", 1.1, 1.0)
  bmi_risk <- ifelse(bmi > 25, 1.3, 1.0)

  round(age_risk * sex_risk * bmi_risk, 2)
}

# YOUR TASK: Use pmap_dbl() to apply the function
risk_scores <- pmap_dbl("YOUR CODE HERE")
cat("Risk scores:\n")
print(risk_scores)

# TASK 5.4: Process multiple datasets with map()
# Create a function to process each study
process_study_data <- function(data) {
  data %>%
    mutate(
      BMI = WEIGHT / (1.70^2),  # Assume height = 170cm
      AGE_GROUP = case_when(
        AGE < 40 ~ "Young",
        AGE >= 40 & AGE < 60 ~ "Middle",
        AGE >= 60 ~ "Senior"
      ),
      ELDERLY_FLAG = ifelse(AGE >= 65, "Y", "N")
    )
}

# YOUR TASK: Apply the function to all studies using map()
processed_studies <- study_datasets %>%
  map("YOUR CODE HERE")

cat("Processed studies summary:\n")
iwalk(processed_studies, ~ {
  cat("Study", .y, "- N:", nrow(.x), "- Mean BMI:", round(mean(.x$BMI), 1), "\n")
})

# TASK 5.5: Extract summary information with map()
# Extract elderly counts from each processed study
elderly_counts <- processed_studies %>%
  map_int("YOUR CODE HERE")  # Extract count of elderly subjects per study

cat("Elderly subjects per study:\n")
print(elderly_counts)

# ===========================
# EXERCISE 6: Predicate Functions and Advanced Patterns
# ===========================

cat("\n=== Exercise 6: Predicate Functions and Advanced Patterns ===\n")

# Lab results for predicate function practice
lab_results <- list(
  glucose = c(90, 95, 110, 85, 92),
  creatinine = c(1.1, 0.9, 1.3, 1.0, 1.2),
  invalid_test = c(NA, NA, NA, NA, NA),
  hemoglobin = c(13.5, 12.1, 14.2, 13.8, 12.9),
  empty_test = numeric(0),
  protein = c(7.2, 6.8, 7.5, 7.1, 6.9)
)

# TASK 6.1: Use keep() and discard() for filtering
# YOUR TASKS:
valid_labs <- lab_results %>%
  keep("YOUR CODE HERE")  # Keep non-empty tests with at least one non-NA value

abnormal_labs <- lab_results %>%
  keep("YOUR CODE HERE")  # Keep tests with values > 100 or < 0.5

cat("Valid lab tests:", names(valid_labs), "\n")
cat("Labs with abnormal values:", names(abnormal_labs), "\n")

# TASK 6.2: Use reduce() for cumulative operations
daily_enrollment <- c(3, 2, 5, 1, 4, 2, 3, 6, 2, 1)

# YOUR TASK: Calculate cumulative enrollment using accumulate()
cumulative_enrollment <- daily_enrollment %>%
  accumulate("YOUR CODE HERE")

cat("Daily enrollment:", daily_enrollment, "\n")
cat("Cumulative enrollment:", cumulative_enrollment, "\n")

# TASK 6.3: Combine datasets using reduce()
enrollment_batches <- list(
  batch1 = tibble(USUBJID = c("001", "002"), SITE = "Site A"),
  batch2 = tibble(USUBJID = c("003", "004", "005"), SITE = "Site B"),
  batch3 = tibble(USUBJID = c("006", "007"), SITE = "Site C")
)

# YOUR TASK: Combine all batches using reduce()
all_subjects <- enrollment_batches %>%
  reduce("YOUR CODE HERE")

cat("Combined enrollment:\n")
print(all_subjects)

# TASK 6.4: Use walk() for side effects
create_site_report <- function(data, site_name) {
  cat("=== Site", site_name, "Report ===\n")
  cat("Subjects enrolled:", nrow(data), "\n")
  cat("Subject IDs:", paste(data$USUBJID, collapse = ", "), "\n\n")
}

# YOUR TASK: Use walk2() to create reports for each batch
walk2("YOUR CODE HERE", "YOUR CODE HERE", create_site_report)

# ===========================
# EXERCISE 7: Error Handling and Function Factories
# ===========================

cat("\n=== Exercise 7: Error Handling and Function Factories ===\n")

# TASK 7.1: Function factories - functions that create functions
# Create a function factory for domain-specific validators

create_domain_checker <- function(domain_prefix, min_length = 5) {
  # YOUR CODE HERE
  # Return a function that:
  # 1. Checks if subject_id has minimum length
  # 2. Checks if subject_id starts with domain_prefix
  # Use str_detect() with regex pattern
}

# Create specific checkers
check_ae_subject <- create_domain_checker("AE")
check_dm_subject <- create_domain_checker("DM")

# Test your function factory
test_subjects <- c("AE-001", "DM-002", "VS-003", "AE-004")
cat("AE subject validation:", map_lgl(test_subjects, check_ae_subject), "\n")

# TASK 7.2: Error handling with possibly() and safely()
risky_calculation <- function(x) {
  if (x < 0) stop("Value must be positive")
  if (x > 100) stop("Value too large")
  sqrt(x)
}

# YOUR TASK: Create safe versions of the function
safe_calc <- possibly("YOUR CODE HERE")  # Use possibly()
safe_calc_detailed <- safely("YOUR CODE HERE")  # Use safely()

test_values <- c(4, 9, -1, 16, 150, 25)

# Test both approaches
safe_results <- map_dbl(test_values, safe_calc)
detailed_results <- map(test_values, safe_calc_detailed)

cat("Safe results:", safe_results, "\n")
cat("First error from safely():\n")
print(detailed_results[[3]])  # Should show the error for -1

# ===========================
# EXERCISE 8: GitHub Copilot in RStudio Practice
# ===========================

cat("\n=== Exercise 8: GitHub Copilot Practice ===\n")

# YOUR TASK: Use GitHub Copilot in RStudio to help create these functions
# Write the comments below in RStudio and let Copilot suggest implementations

# Create function to flag subjects with multiple adverse events in clinical trial


# Function to calculate percent change from baseline for lab values


# Derive safety population flag based on treatment exposure duration


# Function to create CDISC-compliant variable labels for SDTM domains


# Vectorized function to process multiple clinical datasets efficiently


cat("Copilot tips:\n")
cat("1. Use descriptive comments with clinical context\n")
cat("2. Mention specific requirements (CDISC, FDA, etc.)\n")
cat("3. Include expected input/output formats\n")
cat("4. Specify error handling needs\n")

# ===========================
# BONUS CHALLENGE: Complete Clinical Pipeline
# ===========================

cat("\n=== Bonus Challenge: Complete Clinical Data Pipeline ===\n")

# YOUR CHALLENGE: Create a comprehensive data processing function
# This function should combine all R4DS concepts you've learned

create_clinical_pipeline <- function(raw_demo_data, raw_ae_data,
                                   age_cutoff = 65,
                                   output_format = "summary") {
  # YOUR CODE HERE
  # Combine concepts from:
  # 1. Functions (Ch. 19): Proper function structure, arguments, validation
  # 2. Vectors (Ch. 20): Safe type handling and coercion
  # 3. Iteration (Ch. 21): Use map() or for loops for batch processing
  #
  # Pipeline should:
  # - Validate inputs
  # - Clean and standardize data
  # - Derive clinical variables (BMI, age groups, study days)
  # - Create analysis flags
  # - Generate summary statistics
  # - Handle errors gracefully
  # - Return different formats based on output_format parameter
}

# Test your comprehensive pipeline
# final_results <- create_clinical_pipeline(demo, ae_data)

# ===========================
# EXERCISE COMPLETE!
# ===========================

cat("\n🎉 Module 5 Exercise Complete!\n")
cat("R4DS Integration Accomplished:\n")
cat("✓ Functions (Ch. 19): Creation, components, arguments, validation\n")
cat("✓ Vectors (Ch. 20): Types, coercion, subsetting, safe operations\n")
cat("✓ Iteration (Ch. 21): for loops, while loops, map() family\n")
cat("\nClinical Programming Skills Developed:\n")
cat("✓ SAS macro translation to modern R functions\n")
cat("✓ Vectorized operations for clinical data\n")
cat("✓ Error handling and robust function design\n")
cat("✓ Functional programming for batch processing\n")
cat("✓ Advanced patterns: function factories, predicate functions\n")
cat("✓ GitHub Copilot integration for AI-assisted development\n")
cat("\nExcellent work! You're ready for advanced clinical programming!\n")
